#!/bin/sh
# This file is to be sourced from the carp up/down/add-scripts
# It should work in sh, ksh and bash
# Rune Saetre, 2006.03.08



__SCRIPT="$0"

function _warn
{
  echo "${__SCRIPT}: $*" 1>&2
}

function _fail
{
  echo "${__SCRIPT}: $*" 1>&2
  exit 1
}

function _is_alnum
{
  typeset N="$1"
  [ -z "$N" ] && return 1;
  while [ "$N" != "${N#[A-Za-z0-9]}" ]; do
    N="${N#?}"
    [ -z "$N" ] && return 0
  done

  return 1
}

function _is_numeric
{
  typeset V="$1"
  [ -z "$V" ] && return 1;
  while [ "$V" != "${V#[0-9]}" ]; do
    V="${V#?}"
    [ -z "$V" ] && return 0
  done

  return 1
}

function _verify_ip
{
  # Specify IP as argument
  # Returns 0 for valid IP addresses, 1 for invalid ones
  typeset IP="$1" T1="$1." T2 i
  for i in 1 2 3 4; do 
    T2="${T1%%.*}"
    [ "$T2" = "$T1" ] && {
      _warn "Not enough dots in IP address $IP"
      return 1
    }
    T1="${T1#*.}"

    _is_numeric "$T2" || {
      _warn "Non-numeric IP address $IP"
      return 1
    }
  done  
  [ -n "$T1" ] && {
    _warn "Leftover junk in IP address: $IP"
    return 1
  }
  return 0
}


function _get_carp_if
{
  # Takes IP address as argument
  # Leaves if name in $CARP_IF
  if [ -z "$1" ]; then
    _warn "_get_carp_if: Must specify IP address"
    exit 1
  fi

  # Get interface list
  typeset IF_LIST=""
  typeset IF_CFG
  while read CARP_IF; do IF_LIST="${CARP_IF%:*} $IF_LIST"; done <<EOF
$(ifconfig | grep -E '^carp[0-9]+')
EOF

  # Check for one with matching IP
  for CARP_IF in $IF_LIST; do
    if IF_CFG="`ifconfig $CARP_IF 2>/dev/null`" && \
       [ "$IF_CFG" != "${IF_CFG##*inet $1 }" ]; then
      return 0
    fi
  done
  CARP_IF=""
  return 1
}


function _carp_up
{
  # Requires IP address as argument
  _get_carp_if $1 || { _warn "_carp_up: bailing out"; exit 1; }
  echo "$__SCRIPT: Bringing up $CARP_IF ($1)"
  ifconfig $CARP_IF up && 
    echo "$__SCRIPT: $CARP_IF brought up" ||
    _warn "FAILED to bring up $CARP_IF"
}


function _carp_down
{
  # Requires IP address as argument
  _get_carp_if $1 || { _warn "_carp_down: bailing out"; exit 1; }
  echo "$__SCRIPT: Taking $CARP_IF down ($1)"
  ifconfig $CARP_IF down && 
    echo "$__SCRIPT: $CARP_IF brought down" ||
    _warn "FAILED to bring down $CARP_IF"
}

function _carp_up_all
{
  # do ifconfig up on all known carp interfaces
  typeset IF
  while read IF; do 
    IF="${IF%%:*}"
    echo "Bringing up $IF"
    ifconfig "$IF" up && 
      echo "$__SCRIPT: $IF brought up" ||
      _warn "FAILED to bring up $IF"

  done <<EOF
$(ifconfig | grep -E '^carp[0-9]+')
EOF
}


function _carp_down_all
{
  # do ifconfig down on all known carp interfaces
  typeset IF
  while read IF; do 
    IF="${IF%%:*}"
    echo "Bringing down $IF"
    ifconfig "$IF" down && 
      echo "$__SCRIPT: $IF brought down" ||
      _warn "FAILED to bring down $IF" 
  done <<EOF
$(ifconfig | grep -E '^carp[0-9]+')
EOF
}

function _carp_status_all
{
  # list status for all known carp interfaces
  # we support only 999 carp interfaces
  typeset IF READ_IF JUNK CHK STATUS IP

  for IF in /etc/hostname.carp[0-9] \
            /etc/hostname.carp[0-9][0-9] \
            /etc/hostname.carp[0-9][0-9][0-9] ; do
    if [ -f "$IF" ]; then
      IF="${IF##*.}"
      {
        read READ_IF JUNK
        while read CHK STATUS JUNK && [ "$CHK" != "carp:" ]; do true; done
        while read CHK IP JUNK && [ "$CHK" != "inet" ]; do true; done
      } <<EOF      
$(ifconfig $IF)
EOF
    echo "$STATUS - $READ_IF $IP"
    fi 
  done 
}


function _carp_add_if
{
  # Takes name of a new carp IF as argument
  # Adds the carp interface if there is a valid /etc/hostname.carp* for it.

  typeset IF=$1
  typeset CFGFILE="/etc/hostname.$IF"
  typeset AF IP MASK BCAST JUNK CMD

  [ -n "$IF" ] || _fail "Specify an interface name"
  _is_alnum "$IF" || _fail "Specify an alphanumeric interface name ($IF)"
  [ -f "$CFGFILE" ] || _fail "No config file: $CFGFILE"
  
  # Parse out the command line
  while read AF IP MASK BCAST JUNK; do
    [ -z "$AF" -o "${AF#\#}" != "$AF" ] && continue # comment or empty line

    # We have our config line. Check its validity
    [ "$AF" = "inet" ] || 
      _fail "Only address family inet supported"

    _verify_ip $IP    || _fail "Invalid IP address"
    _verify_ip $MASK  || _fail "Invalid netmask"
    _verify_ip $BCAST || _fail "Invalid broadcast address"

    # Construct command to run
    CMD="ifconfig $IF inet $IP netmask $MASK broadcast $BCAST $JUNK"
    _warn "CMD: $CMD" 
    break
  done < $CFGFILE

  [ -z "CMD" ] && _fail "No usable config in $CFGFILE"

  # Does the interface exist?
  ifconfig $IF > /dev/null 2>&1 && 
    _fail "Interface $IF exists. Aborting"

  # Create interface
  _warn "Creating interface"
  ifconfig $IF create || _fail "create failed"
  $CMD || _fail "CMD failed: $CMD"

  echo "$__SCRIPT: Interface $IF added with IP $IP"
}
