#!/bin/sh
# This file is to be sourced from the haproxy stop/start/restart scripts
# It should work in sh, ksh and bash
# 02.11.2006, Rune Saetre <rune.saetre@netcom-gsm.no>

HAPROXY=/usr/local/sbin/haproxy
HA_CFGDIR=/etc/haproxy
HA_PIDDIR=/var/run
HA_OPTIONS="-n 10240"

__SCRIPT="${0##*/}"

function _warn
{
  echo "${__SCRIPT}: $*" 1>&2
}

function _is_numeric
{
  typeset V="$1"
  [ -z "$V" ] && return 1;
  while [ "$V" != "${V#[0-9]}" ]; do
    V="${V#[0-9]}"
    [ -z "$V" ] && return 0
  done

  return 1
}

function _ha_set_vars
{
  # Takes IP and PORT in $1 and $2 
  # sets the variables HA_PIDFILE, HA_CFGFILE and HA_CMD
  typeset IP="$1"
  typeset PORT="$2"

  if [ -z "$1" -o -z "$2" ]; then
    _warn "_ha_set_var needs IP and PORT specified"
    exit 1
  fi


  # Verify IP address
  typeset T1="$IP." T2 i
  for i in 1 2 3 4; do 
    T2="${T1%%.*}"
    [ "$T2" = "$T1" ] && {
      _warn "_ha_set_vars: Not enough dots in IP address $IP"
      return 1
    }
    T1="${T1#*.}"

    _is_numeric "$T2" || {
      _warn "_ha_set_vars: Non-numeric IP address $IP"
      return 1
    }
  done  
  [ -n "$T1" ] && {
    _warn "_ha_set_vars: Leftover junk in IP address: $IP"
    return 1
  }

  # Verify port number
  _is_numeric "$PORT" || {
    _warn "_ha_set_vars: Port is not numeric: $PORT"
    return 1
  }

  HA_PIDFILE="${HA_PIDDIR}/haproxy-${IP}-${PORT}.pid"
  HA_CFGFILE="${HA_CFGDIR}/haproxy-${IP}-${PORT}.cfg"
  HA_CMD="$HAPROXY -f $HA_CFGFILE $HA_OPTIONS -p $HA_PIDFILE"

}

function _ha_get_pids
{
  # Uses global variables HA_PIDFILE and HA_CMD
  # Sets the variable HA_PIDS
  # Extracts PIDs from HA_PIDFILE, checks if they belong to the correct
  # command line and leaves the matching pids in variable HA_PIDS
  # Returns 0 if any pids, 1 if none

  if [ -z "$HA_PIDFILE" -o -z "$HA_CMD" ]; then
    _warn "_ha_get_pids: HA_PIDFILE and HA_CMD not set"
    return 1
  fi

  HA_PIDS=""
  typeset CHECKPIDS=""
  typeset PROCS="`ps uxaww | grep \"$HAPROXY\"`"

  typeset p cp
  typeset a checkpid c d e f g h i j cmdline

  if [ -f $HA_PIDFILE ]; then
    while read p; do CHECKPIDS="$p $CHECKPIDS"; done < $HA_PIDFILE
  fi

  while read a checkpid c d e f g h i j cmdline; do
    for cp in $CHECKPIDS; do
      if [ "$cp" = "$checkpid" ]; then 
        if [ "${cmdline#$HA_CMD}" != "$cmdline" ]; then
          HA_PIDS="$cp $HA_PIDS";
        else
          echo "PID recycled: PID $checkpid CMD: $cmdline" 1>&2
        fi
      fi
    done
  done <<EOF
$PROCS
EOF
  [ -n "$HA_PIDS" ] || return 1

}

function _ha_status
{
  # Needs IP and PORT as arguments
  _ha_set_vars $1 $2 || { _warn "_ha_status: bailing out"; return 1; }

  if [ ! -f "$HA_CFGFILE" ]; then
    _warn "_ha_status: No config file $HA_CFGFILE"
    return 1
  fi

  _ha_get_pids

  if [ -n "$HA_PIDS" ]; then
    echo "RUNNING - $1:$2 pids: $HA_PIDS"
  else
    echo "STOPPED - $1:$2"
  fi

}

function _ha_start
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Starting $1:$2 ..."

  _ha_set_vars $1 $2 || { _warn "_ha_start: bailing out"; return 1; }

  if [ ! -f "$HA_CFGFILE" ]; then
    _warn "_ha_start: No config file $HA_CFGFILE"
    return 1
  fi

  _ha_get_pids

  if [ -n "$HA_PIDS" ]; then
    _warn "$1:$2 already running"
    return 1;
  fi

  $HA_CMD
  echo "$__SCRIPT: $1:$2 started"
}

function _ha_stop
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Stopping $1:$2 ..."

  _ha_set_vars $1 $2 || { _warn "_ha_stop: bailing out"; return 1; }

  if [ ! -f "$HA_CFGFILE" ]; then
    _warn "_ha_stop: No config file $HA_CFGFILE"
  fi

  _ha_get_pids

  if [ -z "$HA_PIDS" ]; then
    _warn "$1:$2 already stopped"
    return 1
  fi

  # I think SIGUSR1 was supposed to initiate soft stop
  kill -USR1 $HA_PIDS &&
    echo "$__SCRIPT: $1:$2 stopped" || 
    _warn "Stopping $1:$2 FAILED"
}

function _ha_restart
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Restarting $1:$2 ..."

  _ha_set_vars $1 $2 || { _warn "_ha_restart: bailing out"; return 1; }

  if [ ! -f "$HA_CFGFILE" ]; then
    _warn "_ha_restart: No config file $HA_CFGFILE"
    return 1
  fi

  _ha_get_pids

  if [ -n "$HA_PIDS" ]; then
    $HA_CMD -sf $HA_PIDS
    echo "$__SCRIPT: $1:$2 restarted"
  else
    _warn "$1:$2 Not running"
  fi
}

