#!/bin/sh
# This file is to be sourced from the pound-tst stop/start/restart scripts
# It should work in sh, ksh and bash
# 15.3.2007, Rune Saetre <rune.saetre@netcom-gsm.no>

POUND_TST=/usr/local/sbin/pound-tst
PNDTST_CFGDIR=/etc/pound-tst
PNDTST_PIDDIR=/var/run

__SCRIPT="${0##*/}"


function _warn
{
  echo "${__SCRIPT}: $*" 1>&2
}

function _is_numeric
{
  typeset V="$1"
  [ -z "$V" ] && return 1;
  while [ "$V" != "${V#[0-9]}" ]; do
    V="${V#[0-9]}"
    [ -z "$V" ] && return 0
  done

  return 1
}

function _pnd_set_vars
{
  # Takes IP and PORT in $1 and $2 
  # sets the variables PNDTST_PIDFILE, PNDTST_CFGFILE and PNDTST_CMD
  typeset IP="$1"
  typeset PORT="$2"

  if [ -z "$1" -o -z "$2" ]; then
    _pnd_warn "_pnd_set_var needs IP and PORT specified"
    exit 1
  fi

  # Verify IP address
  typeset T1="$IP." T2 i
  for i in 1 2 3 4; do 
    T2="${T1%%.*}"
    [ "$T2" = "$T1" ] && {
      _warn "_pnd_set_vars: Not enough dots in IP address $IP"
      return 1
    }
    T1="${T1#*.}"

    _is_numeric "$T2" || {
      _warn "_pnd_set_vars: Non-numeric IP address $IP"
      return 1
    }
  done  
  [ -n "$T1" ] && {
    _warn "_pnd_set_vars: Leftover junk in IP address: $IP"
    return 1
  }

  # Verify port number
  _is_numeric "$PORT" || {
    _warn "_pnd_set_vars: Port is not numeric: $PORT"
    return 1
  }

  PNDTST_PIDFILE="${PNDTST_PIDDIR}/pound-tst-${IP}-${PORT}.pid"
  PNDTST_CFGFILE="${PNDTST_CFGDIR}/pound-${IP}-${PORT}.cfg"
  PNDTST_CMD="$POUND_TST -v -f $PNDTST_CFGFILE -p $PNDTST_PIDFILE"

}


function _pnd_get_pids
{
  # Uses global variables PNDTST_PIDFILE and PNDTST_CMD
  # Sets the variable PNDTST_PIDS
  # Extracts PIDs from PNDTST_PIDFILE, checks if they belong to the correct
  # command line and leaves the matching pids in variable PNDTST_PIDS
  # Returns 0 if any pids, 1 if none

  if [ -z "$PNDTST_PIDFILE" -o -z "$PNDTST_CMD" ]; then
    _warn "_pnd_get_pids: PNDTST_PIDFILE and PNDTST_CMD not set"
    return 1
  fi

  PNDTST_PIDS=""
  typeset CHECKPIDS=""
  typeset PROCS="`ps uxaww | grep \"$POUND_TST\"`"

  typeset p cp
  typeset a checkpid c d e f g h i j cmdline

  if [ -f $PNDTST_PIDFILE ]; then
    while read p; do CHECKPIDS="$p $CHECKPIDS"; done < $PNDTST_PIDFILE
  fi

  while read a checkpid c d e f g h i j cmdline; do
    for cp in $CHECKPIDS; do
      if [ "$cp" = "$checkpid" ]; then 
        if [ "${cmdline#$PNDTST_CMD}" != "$cmdline" ]; then
          PNDTST_PIDS="$cp $PNDTST_PIDS";
        else
          echo "PID recycled: PID $checkpid CMD: $cmdline" 1>&2
        fi
      fi
    done
  done <<EOF
$PROCS
EOF
  [ -n "$PNDTST_PIDS" ] || return 1
}


function _pnd_status
{
  # Needs IP and PORT as arguments
  _pnd_set_vars $1 $2 || { _warn "_pnd_status: bailing out"; return 1; }

  if [ ! -f "$PNDTST_CFGFILE" ]; then
    _warn "_pnd_status: No config file $PNDTST_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PNDTST_PIDS" ]; then
    echo "RUNNING - $1:$2 pids: $PNDTST_PIDS"
  else
    echo "STOPPED - $1:$2"
  fi
}

function _pnd_start
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Starting $1:$2 ..."

  # This can't harm, could even do some good!
  ulimit -n 8192

  _pnd_set_vars $1 $2 || { _warn "_pnd_start: bailing out"; return 1; }

  if [ ! -f "$PNDTST_CFGFILE" ]; then
    _warn "_pnd_start: No config file $PNDTST_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PNDTST_PIDS" ]; then
    _warn "$1:$2 already running"
    return 1;
  fi

  $PNDTST_CMD
  echo "$__SCRIPT: $1:$2 started"
}

function _pnd_stop
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Stopping $1:$2 ..."

  _pnd_set_vars $1 $2 || { _warn "_pnd_stop: bailing out"; return 1; }

  if [ ! -f "$PNDTST_CFGFILE" ]; then
    _warn "_pnd_stop: No config file $PNDTST_CFGFILE"
  fi

  _pnd_get_pids

  if [ -z "$PNDTST_PIDS" ]; then
    _warn "$1:$2 already stopped"
    return 1
  fi

  kill $PNDTST_PIDS && 
    ( rm $PNDTST_PIDFILE; echo "$__SCRIPT: $1:$2 stopped"; ) || 
    _warn "stopping $1:$2 FAILED"
}

function _pnd_restart
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Restarting $1:$2 ..."

  # This can't harm, could even do some good!
  ulimit -n 8192

  _pnd_set_vars $1 $2 || { _warn "_pnd_restart: bailing out"; return 1; }

  if [ ! -f "$PNDTST_CFGFILE" ]; then
    _warn "_pnd_restart: No config file $HA_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PNDTST_PIDS" ]; then
    kill $PNDTST_PIDS && 
      echo "$__SCRIPT: $1:$2 killed $PNDTST_PIDS" || 
      _warn "$1:$2 kill $PNDTST_PIDS FAILED"
    $PNDTST_CMD  && 
      echo "$__SCRIPT: $1:$2 restarted" ||
      _warn "$1:$2 restart FAILED"

  else
    _warn "$1:$2 Not running"
  fi
}



function _pnd_check_cfg
{
  [ -z "$1" ] && { _warn "_pnd_check_cfg: missing arguments"; exit 1; }

  typeset IP_ADDR="$1"
  typeset CFG  

  for CFG in $PNDTST_CFGDIR/pound-${IP_ADDR}-*.cfg; do
    echo "Testing config ($CFG):"
    $POUND_TST -v -f $CFG -c && echo "OK" || echo "Invalid"
  done
}

