#!/bin/sh
# This file is to be sourced from the pound stop/start/restart scripts
# It should work in sh, ksh and bash
# Rune Saetre, 2006.03.08
# 02.11.2006, Rune Saetre <rune.saetre@netcom-gsm.no> - rewritten

POUND=/usr/local/sbin/pound
PND_CFGDIR=/etc/pound
PND_PIDDIR=/var/run

__SCRIPT="${0##*/}"


function _warn
{
  echo "${__SCRIPT}: $*" 1>&2
}

function _is_numeric
{
  typeset V="$1"
  [ -z "$V" ] && return 1;
  while [ "$V" != "${V#[0-9]}" ]; do
    V="${V#[0-9]}"
    [ -z "$V" ] && return 0
  done

  return 1
}

function _pnd_set_vars
{
  # Takes IP and PORT in $1 and $2 
  # sets the variables PND_PIDFILE, PND_CFGFILE and PND_CMD
  typeset IP="$1"
  typeset PORT="$2"

  if [ -z "$1" -o -z "$2" ]; then
    _pnd_warn "_pnd_set_var needs IP and PORT specified"
    exit 1
  fi

  # Verify IP address
  typeset T1="$IP." T2 i
  for i in 1 2 3 4; do 
    T2="${T1%%.*}"
    [ "$T2" = "$T1" ] && {
      _warn "_pnd_set_vars: Not enough dots in IP address $IP"
      return 1
    }
    T1="${T1#*.}"

    _is_numeric "$T2" || {
      _warn "_pnd_set_vars: Non-numeric IP address $IP"
      return 1
    }
  done  
  [ -n "$T1" ] && {
    _warn "_pnd_set_vars: Leftover junk in IP address: $IP"
    return 1
  }

  # Verify port number
  _is_numeric "$PORT" || {
    _warn "_pnd_set_vars: Port is not numeric: $PORT"
    return 1
  }

  PND_PIDFILE="${PND_PIDDIR}/pound-${IP}-${PORT}.pid"
  PND_CFGFILE="${PND_CFGDIR}/pound-${IP}-${PORT}.cfg"
  PND_CMD="$POUND -f $PND_CFGFILE -p $PND_PIDFILE"

}


function _pnd_get_pids
{
  # Uses global variables PND_PIDFILE and PND_CMD
  # Sets the variable PND_PIDS
  # Extracts PIDs from PND_PIDFILE, checks if they belong to the correct
  # command line and leaves the matching pids in variable PND_PIDS
  # Returns 0 if any pids, 1 if none

  if [ -z "$PND_PIDFILE" -o -z "$PND_CMD" ]; then
    _warn "_pnd_get_pids: PND_PIDFILE and PND_CMD not set"
    return 1
  fi

  PND_PIDS=""
  typeset CHECKPIDS=""
  typeset PROCS="`ps uxaww | grep \"$POUND\"`"

  typeset p cp
  typeset a checkpid c d e f g h i j cmdline

  if [ -f $PND_PIDFILE ]; then
    while read p; do CHECKPIDS="$p $CHECKPIDS"; done < $PND_PIDFILE
  fi

  while read a checkpid c d e f g h i j cmdline; do
    for cp in $CHECKPIDS; do
      if [ "$cp" = "$checkpid" ]; then 
        if [ "${cmdline#$PND_CMD}" != "$cmdline" ]; then
          PND_PIDS="$cp $PND_PIDS";
        else
          echo "PID recycled: PID $checkpid CMD: $cmdline" 1>&2
        fi
      fi
    done
  done <<EOF
$PROCS
EOF
  [ -n "$PND_PIDS" ] || return 1
}


function _pnd_status
{
  # Needs IP and PORT as arguments
  _pnd_set_vars $1 $2 || { _warn "_pnd_status: bailing out"; return 1; }

  if [ ! -f "$PND_CFGFILE" ]; then
    _warn "_pnd_status: No config file $PND_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PND_PIDS" ]; then
    echo "RUNNING - $1:$2 pids: $PND_PIDS"
  else
    echo "STOPPED - $1:$2"
  fi
}

function _pnd_start
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Starting $1:$2 ..."

  # This can't harm, could even do some good!
  ulimit -n 8192

  _pnd_set_vars $1 $2 || { _warn "_pnd_start: bailing out"; return 1; }

  if [ ! -f "$PND_CFGFILE" ]; then
    _warn "_pnd_start: No config file $PND_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PND_PIDS" ]; then
    _warn "$1:$2 already running"
    return 1;
  fi

  $PND_CMD
  echo "$__SCRIPT: $1:$2 started"
}

function _pnd_stop
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Stopping $1:$2 ..."

  _pnd_set_vars $1 $2 || { _warn "_pnd_stop: bailing out"; return 1; }

  if [ ! -f "$PND_CFGFILE" ]; then
    _warn "_pnd_stop: No config file $PND_CFGFILE"
  fi

  _pnd_get_pids

  if [ -z "$PND_PIDS" ]; then
    _warn "$1:$2 already stopped"
    return 1
  fi

  kill $PND_PIDS && 
    echo "$__SCRIPT: $1:$2 stopped" || 
    _warn "stopping $1:$2 FAILED"
}

function _pnd_restart
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Restarting $1:$2 ..."

  # This can't harm, could even do some good!
  ulimit -n 8192

  _pnd_set_vars $1 $2 || { _warn "_pnd_restart: bailing out"; return 1; }

  if [ ! -f "$PND_CFGFILE" ]; then
    _warn "_pnd_restart: No config file $HA_CFGFILE"
    return 1
  fi

  _pnd_get_pids

  if [ -n "$PND_PIDS" ]; then
    kill $PND_PIDS && 
      echo "$__SCRIPT: $1:$2 killed $PND_PIDS" || 
      _warn "$1:$2 kill $PND_PIDS FAILED"
    $PND_CMD  && 
      echo "$__SCRIPT: $1:$2 restarted" ||
      _warn "$1:$2 restart FAILED"

  else
    _warn "$1:$2 Not running"
  fi
}



function _pnd_check_cfg
{
  [ -z "$1" ] && { _warn "_pnd_check_cfg: missing arguments"; exit 1; }

  typeset IP_ADDR="$1"
  typeset CFG  

  for CFG in $PND_CFGDIR/pound-${IP_ADDR}-*.cfg; do
    echo "Testing config ($CFG):"
    $POUND -v -f $CFG -c && echo "OK" || echo "Invalid"
  done
}

