#!/bin/sh
# This file is to be sourced from the stunnel stop/start/restart scripts
# It should work in sh, ksh and bash
# 02.11.2006, Rune Saetre <rune.saetre@netcom-gsm.no>

STUNNEL=/usr/local/sbin/stunnel
STN_CFGDIR=/etc/stunnel
STN_PIDDIR=/var/run


__SCRIPT="${0##*/}"

function _warn
{
  echo "${__SCRIPT}: $*" 1>&2
}

function _is_numeric
{
  typeset V="$1"
  [ -z "$V" ] && return 1;
  while [ "$V" != "${V#[0-9]}" ]; do
    V="${V#[0-9]}"
    [ -z "$V" ] && return 0
  done

  return 1
}

function _stn_set_vars
{
  # Takes IP and PORT in $1 and $2 
  # sets the variables STN_PIDFILE, STN_CFGFILE and STN_CMD
  typeset IP="$1"
  typeset PORT="$2"

  if [ -z "$1" -o -z "$2" ]; then
    _warn "_stn_set_var needs IP and PORT specified"
    exit 1
  fi


  # Verify IP address
  typeset T1="$IP." T2 i
  for i in 1 2 3 4; do 
    T2="${T1%%.*}"
    [ "$T2" = "$T1" ] && {
      _warn "_stn_set_vars: Not enough dots in IP address $IP"
      return 1
    }
    T1="${T1#*.}"

    _is_numeric "$T2" || {
      _warn "_stn_set_vars: Non-numeric IP address $IP"
      return 1
    }
  done  
  [ -n "$T1" ] && {
    _warn "_stn_set_vars: Leftover junk in IP address: $IP"
    return 1
  }

  # Verify port number
  _is_numeric "$PORT" || {
    _warn "_stn_set_vars: Port is not numeric: $PORT"
    return 1
  }

  STN_PIDFILE="${STN_PIDDIR}/stunnel-${IP}-${PORT}.pid"
  STN_CFGFILE="${STN_CFGDIR}/stunnel-${IP}-${PORT}.cfg"
  STN_CMD="$STUNNEL $STN_CFGFILE"

  # Check that the pidfile in the stunnel config file matches the 
  # one we just figured out. If not: issue a warning and die.
  typeset var eq pid
  typeset i=0
  typeset cfg_has_pidfile=false

  while read var eq val; do
    i=$(($i+1))
    if [ "$var" = "pid" ]; then
      if [ "$val" != "$STN_PIDFILE" ]; then
        _warn "$STN_CFGFILE line $i:"
        _warn "  Specified pidfile $var should be $STN_PIDFILE"
        _warn "  Not starting"
        STN_PIDFILE=""
        STN_CFGFILE=""
        STN_CMD=""
        return 1
      else
        cfg_has_pidfile=true
      fi
    fi
  done < $STN_CFGFILE

  if [ "$cfg_has_pidfile" != "true" ]; then
    _warn "$STN_CFGFILE:"
    _warn "  Please specify \"pid = $STN_PIDFILE\""
    _warn "  Not starting"
    STN_PIDFILE=""
    STN_CFGFILE=""
    STN_CMD=""
    return 1
  fi
}

function _stn_get_pids
{
  # Uses global variables STN_PIDFILE and STN_CMD
  # Sets the variable STN_PIDS
  # Extracts PIDs from STN_PIDFILE, checks if they belong to the correct
  # command line and leaves the matching pids in variable STN_PIDS
  # Returns 0 if any pids, 1 if none

  if [ -z "$STN_PIDFILE" -o -z "$STN_CMD" ]; then
    _warn "_stn_get_pids: STN_PIDFILE and STN_CMD not set"
    return 1
  fi

  STN_PIDS=""
  typeset CHECKPIDS=""
  typeset PROCS="`ps uxaww | grep \"$STUNNEL\"`"

  typeset p cp
  typeset a checkpid c d e f g h i j cmdline

  if [ -f $STN_PIDFILE ]; then
    while read p; do CHECKPIDS="$p $CHECKPIDS"; done < $STN_PIDFILE
  fi

  while read a checkpid c d e f g h i j cmdline; do
    for cp in $CHECKPIDS; do
      if [ "$cp" = "$checkpid" ]; then 
        if [ "${cmdline#$STN_CMD}" != "$cmdline" ]; then
          STN_PIDS="$cp $STN_PIDS";
        else
          echo "PID recycled: PID $checkpid CMD: $cmdline" 1>&2
        fi
      fi
    done
  done <<EOF
$PROCS
EOF
  [ -n "$STN_PIDS" ] || return 1

}

function _stn_status
{
  # Needs IP and PORT as arguments
  _stn_set_vars $1 $2 || { _warn "_stn_status: bailing out"; return 1; }

  if [ ! -f "$STN_CFGFILE" ]; then
    _warn "_stn_status: No config file $STN_CFGFILE"
    return 1
  fi

  _stn_get_pids

  if [ -n "$STN_PIDS" ]; then
    echo "RUNNING - $1:$2 pids: $STN_PIDS"
  else
    echo "STOPPED - $1:$2"
  fi
}


function _stn_start
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Starting $1:$2 ..."

  # This can't harm to do from time to time
  ulimit -n 8192

  _stn_set_vars $1 $2 || { _warn "_stn_start: bailing out"; return 1; }

  if [ ! -f "$STN_CFGFILE" ]; then
    _warn "_stn_start: No config file $STN_CFGFILE"
    return 1
  fi

  _stn_get_pids

  if [ -n "$STN_PIDS" ]; then
    _warn "$1:$2 already running"
    return 1;
  fi

  $STN_CMD
  echo "$__SCRIPT: $1:$2 started"
}

function _stn_stop
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Stopping $1:$2 ..."

  _stn_set_vars $1 $2 || { _warn "_stn_stop: bailing out"; return 1; }

  if [ ! -f "$STN_CFGFILE" ]; then
    _warn "_stn_stop: No config file $STN_CFGFILE"
  fi

  _stn_get_pids

  if [ -z "$STN_PIDS" ]; then
    _warn "$1:$2 already stopped"
    return 1
  fi

  # I think SIGUSR1 was supposed to initiate soft stop
  kill -USR1 $STN_PIDS &&
    echo "$__SCRIPT: $1:$2 stopped" || 
    _warn "Stopping $1:$2 FAILED"
}

function _stn_restart
{
  # Needs IP and PORT as arguments
  echo "$__SCRIPT: Restarting $1:$2 ..."

  # This can't harm to do from time to time
  ulimit -n 8192

  _stn_set_vars $1 $2 || { _warn "_stn_restart: bailing out"; return 1; }

  if [ ! -f "$STN_CFGFILE" ]; then
    _warn "_stn_restart: No config file $STN_CFGFILE"
    return 1
  fi

  _stn_get_pids

  if [ -n "$STN_PIDS" ]; then
    kill $STN_PIDS && 
      echo "$__SCRIPT: $1:$2 killed $STN_PIDS" || 
      _warn "$1:$2 kill $STN_PIDS FAILED"
    $STN_CMD &&
      echo "$__SCRIPT: $1:$2 restarted" ||
      _warn "$1:$2 restart FAILED"

  else
    _warn "$1:$2 Not running"
  fi
}

